﻿using System;
using System.Windows.Forms;
using CT_NFC_C3x5_Lib;

namespace CT_NFC_C3x5_SDK_Sample
{
    public partial class Sample : Form
    {
        Reader NFCReader = new Reader();

        public Sample()
        {
            InitializeComponent();

            NFCReader.Dev_SetLog(richTextBox_Log);

            GetComport();

            InitMifareSector();
        }
        private void button_Clear_Log_Click(object sender, EventArgs e)
        {
            richTextBox_Log.Clear();
        }

        #region Device Function
        private void GetComport()
        {
            comboBox_Comport.Items.Clear();
            string[] comport_list = NFCReader.Dev_GetPort();
            comboBox_Comport.Items.AddRange(comport_list);

            if (comboBox_Comport.Items.Count > 0)
            {
                comboBox_Comport.SelectedIndex = 0;
            }

            if (comboBox_Baudrate.Items.Count > 0)
            {
                comboBox_Baudrate.SelectedIndex = 0;
            }
        }
        
        private void comboBox_Comport_Click(object sender, EventArgs e)
        {
            GetComport();
        }

        private void button_Device_Open_Click(object sender, EventArgs e)
        {
            if (comboBox_Comport.SelectedIndex == -1)
            {
                MessageBox.Show("No Device Found !");
                return;
            }

            int baudrate = 0;
            if (!int.TryParse(comboBox_Baudrate.SelectedItem.ToString(), out baudrate))
            {
                MessageBox.Show("BaudRate Error !");
                return;
            }

            NFCReader.Dev_OpenWithBR(comboBox_Comport.SelectedItem.ToString(), baudrate);
        }

        private void button_Device_Close_Click(object sender, EventArgs e)
        {
            NFCReader.Dev_Close();
        }
        #endregion

        #region ISO14443A Function
        private void button_ISO14443A_Config_Click(object sender, EventArgs e)
        {
            NFCReader.ISO14443A_Config();
        }

        private void button_GetTagInfo_Click(object sender, EventArgs e)
        {
            ClearTagInfo();

            Result.ISO14443A.Request req_obj = new Result.ISO14443A.Request();
            if (NFCReader.ISO14443A_Request(req_obj) == Reader.ResponseFlag.Operation_success)
            {
                if (NFCReader.ISO14443A_Halt() == Reader.ResponseFlag.Operation_success)
                {
                    Result.ISO14443A.Req_Anti_Select obj = new Result.ISO14443A.Req_Anti_Select();
                    if (NFCReader.ISO14443A_Req_Anti_Select(Setup.ISO14443A.ReqMode.WakeupCmd, Setup.ISO14443A.CollisionMarkValue.MaskZero, obj) == Reader.ResponseFlag.Operation_success)
                    {
                        textBox_UID.Text = BitConverter.ToString(obj.UID).Replace("-", "");
                        textBox_ATQA.Text = BitConverter.ToString(req_obj.ATQA).Replace("-", "");
                        textBox_SAK.Text = Convert.ToString(obj.SAK, 16).ToUpper().PadLeft(2, '0');
                    }
                }
            }
        }

        private void ClearTagInfo()
        {
            textBox_UID.Text = "";
            textBox_ATQA.Text = "";
            textBox_SAK.Text = "";
        }
        #endregion

        #region NTag Function
        private void button_UltraLight_Read_Click(object sender, EventArgs e)
        {
            if (string.IsNullOrEmpty(textBox_UltraLight_Read_Block_Num.Text))
            {
                MessageBox.Show("UltraLight Read Block Num Error !");
                return;
            }

            listView_UltraLight_Read_Data.Items.Clear();

            byte blockNum = Convert.ToByte(textBox_UltraLight_Read_Block_Num.Text, 16);
            Result.MifareUltralight.ReadBlock obj = new Result.MifareUltralight.ReadBlock();

            if (NFCReader.MifareUltralight_ReadBlock(blockNum, obj) == Reader.ResponseFlag.Operation_success)
            {
                for (int i = 0; i < 4; i++)
                {
                    string Num = (blockNum + i).ToString("X").PadLeft(2, '0');
                    string Data = BitConverter.ToString(obj.Block16Bytes, i * 4, 4).Replace("-", "");

                    ListViewItem item = new ListViewItem(Num);
                    item.SubItems.Add(Data);
                    listView_UltraLight_Read_Data.Items.Add(item);
                }
            }
        }

        private void button_UltraLight_Write_Click(object sender, EventArgs e)
        {
            if (string.IsNullOrEmpty(textBox_UltraLight_Write_Block_Num.Text))
            {
                MessageBox.Show("UltraLight Write Block Num Error !");
                return;
            }

            byte blockNum = Convert.ToByte(textBox_UltraLight_Write_Block_Num.Text, 16);
            byte[] data = ByteUtility.HexToByte(textBox_UltraLight_Write_Data.Text.Replace("-", ""));

            NFCReader.MifareUltralight_WriteBlock(blockNum, data);
        }
        #endregion

        #region Mifare Function
        private void InitMifareSector()
        {
            for (int i = 0; i <= 15; i++)
            {
                comboBox_Mifare_Sector.Items.Add(i);
            }

            if (comboBox_Mifare_Sector.Items.Count > 0)
            {
                comboBox_Mifare_Sector.SelectedIndex = 0;
            }

            SetMifareBlockNum(comboBox_Mifare_Sector.SelectedIndex);
        }

        private void SetMifareBlockNum(int sector_num)
        {
            comboBox_Mifare_Block.Items.Clear();

            for (int i = 0; i <= 3; i++)
            {
                comboBox_Mifare_Block.Items.Add((4 * sector_num) + i);
            }

            if (comboBox_Mifare_Block.Items.Count > 0)
            {
                comboBox_Mifare_Block.SelectedIndex = 0;
            }
        }

        private void comboBox_Mifare_Sector_SelectedIndexChanged(object sender, EventArgs e)
        {
            SetMifareBlockNum(comboBox_Mifare_Sector.SelectedIndex);
        }

        private void button_Mifare_Load_Key_Click(object sender, EventArgs e)
        {
            if (string.IsNullOrEmpty(textBox_Mifare_Key.Text))
            {
                MessageBox.Show("Mifare Key Error !");
                return;
            }

            byte[] key = ByteUtility.HexToByte(textBox_Mifare_Key.Text.Replace("-", ""));

            NFCReader.Mifare_Loadkey(key);
        }

        private void button_Mifare_Authentication_Click(object sender, EventArgs e)
        {
            if (string.IsNullOrEmpty(textBox_UID.Text))
            {
                MessageBox.Show("Mifare UID Error !");
                return;
            }

            Setup.Mifare.SelectKey selectKey = radioButton_Mifare_KeyA.Checked ? Setup.Mifare.SelectKey.KeyA : Setup.Mifare.SelectKey.KeyB;
            byte blockNum = Convert.ToByte(comboBox_Mifare_Block.SelectedItem);            
            byte[] uid = ByteUtility.HexToByte(textBox_UID.Text);

            NFCReader.Mifare_Authentication(selectKey, blockNum, uid);
        }

        private void button_Mifare_Read_Click(object sender, EventArgs e)
        {
            textBox_Mifare_Read_Data.Text = "";

            byte blockNum = Convert.ToByte(comboBox_Mifare_Block.SelectedItem);
            Result.Mifare.ReadBlock obj = new Result.Mifare.ReadBlock();

            if (NFCReader.Mifare_ReadBlock(blockNum, obj) == Reader.ResponseFlag.Operation_success)
            {
                textBox_Mifare_Read_Data.Text = BitConverter.ToString(obj.Block16Bytes).Replace("-", "");
            }
        }

        private void button_Mifare_Write_Click(object sender, EventArgs e)
        {
            byte blockNum = Convert.ToByte(comboBox_Mifare_Block.SelectedItem);
            byte[] data = ByteUtility.HexToByte(textBox_Mifare_Write_Data.Text.Replace("-", ""));

            NFCReader.Mifare_WriteBlock(blockNum, data);
        }
        #endregion
    }
}
